/*
 * Decompiled with CFR 0.152.
 */
package com.github.xingshuangs.iot.protocol.rtp.service;

import com.github.xingshuangs.iot.common.buff.ByteWriteBuff;
import com.github.xingshuangs.iot.protocol.rtp.enums.EH264NaluType;
import com.github.xingshuangs.iot.protocol.rtp.enums.EH264SliceType;
import com.github.xingshuangs.iot.protocol.rtp.model.RtpPackage;
import com.github.xingshuangs.iot.protocol.rtp.model.frame.H264VideoFrame;
import com.github.xingshuangs.iot.protocol.rtp.model.frame.RawFrame;
import com.github.xingshuangs.iot.protocol.rtp.model.payload.H264NaluBuilder;
import com.github.xingshuangs.iot.protocol.rtp.model.payload.H264NaluFuA;
import com.github.xingshuangs.iot.protocol.rtp.model.payload.H264NaluFuHeader;
import com.github.xingshuangs.iot.protocol.rtp.model.payload.H264NaluHeader;
import com.github.xingshuangs.iot.protocol.rtp.model.payload.H264NaluSingle;
import com.github.xingshuangs.iot.protocol.rtp.service.IPayloadParser;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class H264VideoParser
implements IPayloadParser {
    private static final Logger log = LoggerFactory.getLogger(H264VideoParser.class);
    private final Integer payloadNumber;
    private long baseTimestamp = 0L;
    private H264VideoFrame lastFrame;
    private final List<H264VideoFrame> cacheFrameList = new ArrayList<H264VideoFrame>();
    private Consumer<RawFrame> frameHandle;
    private final List<RtpPackage> rtpPackageList = new ArrayList<RtpPackage>();
    private final List<RtpPackage> naluBuffers = new ArrayList<RtpPackage>();
    private RtpPackage lastRtpPackage;
    private boolean hasBFrame;

    public H264VideoParser(Integer payloadNumber) {
        this.payloadNumber = payloadNumber;
    }

    private void resetBuffers() {
        this.naluBuffers.clear();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private H264VideoFrame doRtpNaluSingleBuffers() {
        if (this.naluBuffers.isEmpty()) {
            return null;
        }
        try {
            this.naluBuffers.sort(Comparator.comparingInt(a -> a.getHeader().getSequenceNumber()));
            boolean match = this.matchLostNumber();
            if (!match) {
                H264VideoFrame h264VideoFrame = null;
                return h264VideoFrame;
            }
            RtpPackage rtp = this.naluBuffers.get(0);
            EH264NaluType currentNaluType = this.queryNaluType();
            List<byte[]> naluSingleBytes = this.extractNaluSingleBytes();
            if (naluSingleBytes.isEmpty()) {
                H264VideoFrame h264VideoFrame = null;
                return h264VideoFrame;
            }
            int sum = naluSingleBytes.stream().mapToInt(x -> ((byte[])x).length).sum() + (naluSingleBytes.size() - 1) * 4;
            ByteWriteBuff buff = new ByteWriteBuff(sum);
            for (int i = 0; i < naluSingleBytes.size() - 1; ++i) {
                buff.putBytes(naluSingleBytes.get(i));
                buff.putBytes(new byte[]{0, 0, 0, 1});
            }
            buff.putBytes(naluSingleBytes.get(naluSingleBytes.size() - 1));
            H264VideoFrame h264VideoFrame = new H264VideoFrame(currentNaluType, rtp.getHeader().getTimestamp() - this.baseTimestamp, buff.getData());
            return h264VideoFrame;
        }
        finally {
            this.naluBuffers.clear();
        }
    }

    private EH264NaluType queryNaluType() {
        for (RtpPackage rtpPackage : this.naluBuffers) {
            H264NaluHeader h264NaluHeader = H264NaluHeader.fromBytes(rtpPackage.getPayload());
            if (h264NaluHeader.getType() == EH264NaluType.NON_IDR_SLICE || h264NaluHeader.getType() == EH264NaluType.IDR_SLICE) {
                return h264NaluHeader.getType();
            }
            if (h264NaluHeader.getType() != EH264NaluType.FU_A) continue;
            H264NaluFuA h264NaluFuA = (H264NaluFuA)H264NaluBuilder.parsePackage(rtpPackage.getPayload());
            return h264NaluFuA.getFuHeader().getType();
        }
        return EH264NaluType.NON_IDR_SLICE;
    }

    private List<byte[]> extractNaluSingleBytes() {
        ArrayList<byte[]> naluSingleBytes = new ArrayList<byte[]>();
        ArrayList<H264NaluFuA> naluFuAList = new ArrayList<H264NaluFuA>();
        for (RtpPackage rtpPackage : this.naluBuffers) {
            H264NaluHeader h264NaluHeader = H264NaluHeader.fromBytes(rtpPackage.getPayload());
            if (h264NaluHeader.getType() == EH264NaluType.NON_IDR_SLICE || h264NaluHeader.getType() == EH264NaluType.IDR_SLICE) {
                naluSingleBytes.add(rtpPackage.getPayload());
                continue;
            }
            if (h264NaluHeader.getType() != EH264NaluType.FU_A) continue;
            H264NaluFuA h264NaluFuA = (H264NaluFuA)H264NaluBuilder.parsePackage(rtpPackage.getPayload());
            if (h264NaluFuA.getFuHeader().isStart()) {
                naluFuAList.clear();
            }
            naluFuAList.add(h264NaluFuA);
            if (!h264NaluFuA.getFuHeader().isEnd()) continue;
            int sum = naluFuAList.stream().mapToInt(x -> x.getPayload().length).sum();
            ByteWriteBuff buff = new ByteWriteBuff(sum);
            naluFuAList.forEach(x -> buff.putBytes(x.getPayload()));
            H264NaluSingle single = new H264NaluSingle();
            single.getHeader().setForbiddenZeroBit(h264NaluFuA.getHeader().isForbiddenZeroBit());
            single.getHeader().setNri(h264NaluFuA.getHeader().getNri());
            single.getHeader().setType(h264NaluFuA.getFuHeader().getType());
            single.setPayload(buff.getData());
            naluSingleBytes.add(single.toByteArray());
        }
        return naluSingleBytes;
    }

    private boolean matchLostNumber() {
        int lostNumber = 0;
        for (int i = 1; i < this.naluBuffers.size(); ++i) {
            if (this.naluBuffers.get(i).getHeader().getSequenceNumber() - this.naluBuffers.get(i - 1).getHeader().getSequenceNumber() == 1) continue;
            ++lostNumber;
        }
        int idrSliceMinNumber = 2;
        int nonIdrSliceMinNumber = 4;
        RtpPackage rtp = this.naluBuffers.get(0);
        H264NaluHeader naluHeader = H264NaluHeader.fromBytes(rtp.getPayload());
        if (naluHeader.getType() == EH264NaluType.IDR_SLICE && lostNumber > idrSliceMinNumber) {
            log.debug("When a Single NALU is processed, data sequence numbers are discontinuous, resulting in key frame data loss due to packet loss. The frame data is discarded. The total number of [{}] and the number of lost [{}] exceeds [{}].", new Object[]{this.naluBuffers.size(), lostNumber, idrSliceMinNumber});
            return false;
        }
        if (naluHeader.getType() == EH264NaluType.NON_IDR_SLICE && lostNumber > nonIdrSliceMinNumber) {
            log.debug("When a Single NALU is processed, data sequence numbers are discontinuous, resulting in non-key frame data loss due to packet loss. The frame data is discarded. The total number of [{}] and the number of lost [{}] exceeds [{}].", new Object[]{this.naluBuffers.size(), lostNumber, nonIdrSliceMinNumber});
            return false;
        }
        if (naluHeader.getType() == EH264NaluType.FU_A) {
            H264NaluFuHeader naluFuHeader = H264NaluFuHeader.fromBytes(rtp.getPayload(), 1);
            if (naluFuHeader.getType() == EH264NaluType.IDR_SLICE && lostNumber > idrSliceMinNumber) {
                log.debug("When processing the NALU of FUA, the data sequence number is discontinuous, resulting in the data loss of key frame data due to packet loss. The frame data is discarded. The total number of [{}] and the number of lost [{}] exceed [{}].", new Object[]{this.naluBuffers.size(), lostNumber, idrSliceMinNumber});
                return false;
            }
            if (naluFuHeader.getType() == EH264NaluType.NON_IDR_SLICE && lostNumber > nonIdrSliceMinNumber) {
                log.debug("When processing the NALU of FUA, the data sequence number is discontinuous, resulting in the data loss of non-key frame data due to packet loss. The frame data is discarded. The total number of [{}] and the number of lost [{}] exceed [{}].", new Object[]{this.naluBuffers.size(), lostNumber, nonIdrSliceMinNumber});
                return false;
            }
        } else if (naluHeader.getType() != EH264NaluType.IDR_SLICE && naluHeader.getType() != EH264NaluType.NON_IDR_SLICE) {
            log.error("The data type of NALU in RTP cannot be recognized, the frame data is discarded, frame type [{}]", (Object)naluHeader.getType());
            return false;
        }
        return true;
    }

    @Override
    public void processPackage(RtpPackage rtpPackage) {
        if (rtpPackage.getHeader().getPayloadType() != this.payloadNumber.intValue()) {
            log.warn("payload numbers are inconsistent, expect[{}], actual[{}], ignore this message.", (Object)this.payloadNumber, (Object)rtpPackage.getHeader().getPayloadType());
            return;
        }
        RtpPackage rtp = this.addLastRtpPackage(rtpPackage);
        if (rtp == null) {
            return;
        }
        if (this.baseTimestamp == 0L) {
            this.baseTimestamp = rtp.getHeader().getTimestamp();
        }
        H264NaluHeader header = H264NaluHeader.fromBytes(rtp.getPayload());
        switch (header.getType()) {
            case AUD: {
                this.resetBuffers();
                break;
            }
            case SEI: 
            case PPS: 
            case SPS: {
                H264VideoFrame frame = new H264VideoFrame(header.getType(), rtp.getHeader().getTimestamp() - this.baseTimestamp, rtp.getPayload());
                this.videoFrameHandle(frame);
                break;
            }
            case NON_IDR_SLICE: 
            case IDR_SLICE: 
            case FU_A: {
                this.naluBuffers.add(rtp);
                if (!rtp.getHeader().isMarker()) break;
                H264VideoFrame frame = this.doRtpNaluSingleBuffers();
                this.videoFrameHandle(frame);
                break;
            }
            case STAP_A: 
            case STAP_B: {
                break;
            }
            default: {
                log.error("RTP parsing unknown data type [{}], timestamp [{}]", (Object)header.getType(), (Object)rtp.getHeader().getTimestamp());
            }
        }
    }

    private RtpPackage addLastRtpPackage(RtpPackage rtp) {
        RtpPackage currentRtp = null;
        this.rtpPackageList.add(rtp);
        this.rtpPackageList.sort(Comparator.comparingInt(a -> a.getHeader().getSequenceNumber()));
        if (this.rtpPackageList.size() > 5) {
            currentRtp = this.rtpPackageList.remove(0);
        }
        if (currentRtp != null && this.lastRtpPackage != null && this.lastRtpPackage.getHeader().getSequenceNumber() > currentRtp.getHeader().getSequenceNumber()) {
            log.debug("The sequence number is not always ascending when receiving RTP data");
        }
        this.lastRtpPackage = currentRtp;
        return currentRtp;
    }

    @Override
    public void onFrameHandle(Consumer<RawFrame> frameHandle) {
        this.frameHandle = frameHandle;
    }

    private void videoFrameHandle(H264VideoFrame frame) {
        H264VideoFrame h264VideoFrame;
        if (this.frameHandle == null || frame == null || frame.getPts() < 0L) {
            return;
        }
        if (frame.getNaluType() == EH264NaluType.IDR_SLICE || frame.getNaluType() == EH264NaluType.NON_IDR_SLICE) {
            h264VideoFrame = this.dtsHandle(frame);
            this.addLastFrame(frame);
            if (h264VideoFrame == null) {
                return;
            }
        } else {
            h264VideoFrame = frame;
        }
        try {
            this.frameHandle.accept(h264VideoFrame);
        }
        catch (Exception e) {
            log.error(e.getMessage(), (Throwable)e);
        }
    }

    private H264VideoFrame dtsHandle(H264VideoFrame frame) {
        if (frame.getSliceType() == EH264SliceType.B && !this.hasBFrame) {
            this.hasBFrame = true;
        }
        if (this.lastFrame == null) {
            return null;
        }
        if (this.hasBFrame && frame.getSliceType() != EH264SliceType.I && this.cacheFrameList.size() >= 5) {
            long delta = (this.cacheFrameList.get(4).getPts() - this.cacheFrameList.get(0).getPts()) / 4L;
            frame.setDts(this.lastFrame.getDts() + delta);
        }
        this.lastFrame.setDuration((int)(frame.getDts() - this.lastFrame.getDts()));
        if (this.lastFrame.getDuration() < 0) {
            this.lastFrame.setDuration(0);
            if (frame.getSliceType() != EH264SliceType.I) {
                frame.setDts(this.lastFrame.getDts());
            } else {
                this.lastFrame.setDts(frame.getDts());
            }
        }
        return this.lastFrame;
    }

    private void addLastFrame(H264VideoFrame frame) {
        this.lastFrame = frame;
        this.cacheFrameList.add(frame);
        this.cacheFrameList.sort((a, b) -> (int)(a.getTimestamp() - b.getTimestamp()));
        if (this.cacheFrameList.size() > 10) {
            this.cacheFrameList.remove(0);
        }
    }
}

