/*
 * Decompiled with CFR 0.152.
 */
package com.github.xingshuangs.iot.protocol.s7.serializer;

import com.github.xingshuangs.iot.common.buff.ByteReadBuff;
import com.github.xingshuangs.iot.common.buff.ByteWriteBuff;
import com.github.xingshuangs.iot.common.enums.EDataType;
import com.github.xingshuangs.iot.common.serializer.IPLCSerializable;
import com.github.xingshuangs.iot.exceptions.S7CommException;
import com.github.xingshuangs.iot.protocol.s7.enums.EPlcType;
import com.github.xingshuangs.iot.protocol.s7.model.DataItem;
import com.github.xingshuangs.iot.protocol.s7.model.RequestItem;
import com.github.xingshuangs.iot.protocol.s7.serializer.S7Parameter;
import com.github.xingshuangs.iot.protocol.s7.serializer.S7ParseData;
import com.github.xingshuangs.iot.protocol.s7.serializer.S7Variable;
import com.github.xingshuangs.iot.protocol.s7.service.S7PLC;
import com.github.xingshuangs.iot.protocol.s7.utils.AddressUtil;
import java.lang.reflect.Field;
import java.nio.charset.Charset;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

public class S7Serializer
implements IPLCSerializable {
    private final S7PLC s7PLC;
    private Charset charsets;

    public S7Serializer(S7PLC s7PLC) {
        this.s7PLC = s7PLC;
        this.charsets = Charset.forName("GB2312");
    }

    public static S7Serializer newInstance(S7PLC s7PLC) {
        return new S7Serializer(s7PLC);
    }

    private List<S7ParseData> parseBean(Class<?> targetClass) {
        ArrayList<S7ParseData> s7ParseDataList = new ArrayList<S7ParseData>();
        for (Field field : targetClass.getDeclaredFields()) {
            S7Variable s7Variable = field.getAnnotation(S7Variable.class);
            if (s7Variable == null) continue;
            S7Parameter parameter = new S7Parameter(s7Variable.address(), s7Variable.type(), s7Variable.count());
            S7ParseData s7ParseData = this.createS7ParseData(parameter, field);
            s7ParseDataList.add(s7ParseData);
        }
        return s7ParseDataList;
    }

    private List<S7ParseData> parseBean(List<S7Parameter> parameters) {
        try {
            ArrayList<S7ParseData> s7ParseDataList = new ArrayList<S7ParseData>();
            for (S7Parameter p : parameters) {
                if (p == null) {
                    throw new S7CommException("null exists in the parameters list");
                }
                S7ParseData s7ParseData = this.createS7ParseData(p, p.getClass().getDeclaredField("value"));
                s7ParseDataList.add(s7ParseData);
            }
            return s7ParseDataList;
        }
        catch (NoSuchFieldException e) {
            throw new S7CommException(e);
        }
    }

    private void checkS7Variable(S7Parameter parameter) {
        if (parameter.getAddress().isEmpty()) {
            throw new S7CommException("[address] in the S7 parameter cannot be empty");
        }
        if (parameter.getCount() < 0) {
            throw new S7CommException("[count] in the S7 parameter cannot be negative");
        }
        if (parameter.getDataType() == EDataType.STRING && parameter.getCount() > 254) {
            throw new S7CommException("The [count] value of the string type in the S7 parameter cannot be greater than 254");
        }
        if (parameter.getDataType() != EDataType.BYTE && parameter.getDataType() != EDataType.STRING && parameter.getCount() > 1) {
            throw new S7CommException("In the S7 parameter, only [type]= bytes and [count] of string type data can be greater than 1, and the rest must be equal to 1");
        }
    }

    private void readDataByCondition(List<S7ParseData> s7ParseDataList) {
        if (s7ParseDataList.isEmpty()) {
            throw new S7CommException("The number of parsed annotation data is empty, and the data cannot be read");
        }
        List<RequestItem> requestItems = s7ParseDataList.stream().map(S7ParseData::getRequestItem).collect(Collectors.toList());
        List<DataItem> dataItems = this.s7PLC.readS7Data(requestItems);
        if (s7ParseDataList.size() != dataItems.size()) {
            throw new S7CommException("The number of field parsing items required is inconsistent with the number of data items returned");
        }
        for (int i = 0; i < dataItems.size(); ++i) {
            s7ParseDataList.get(i).setDataItem(dataItems.get(i));
        }
    }

    private S7ParseData createS7ParseData(S7Parameter p, Field field) {
        this.checkS7Variable(p);
        S7ParseData s7ParseData = new S7ParseData();
        s7ParseData.setDataType(p.getDataType());
        s7ParseData.setCount(p.getCount());
        s7ParseData.setField(field);
        if (p.getDataType() == EDataType.BOOL) {
            s7ParseData.setRequestItem(AddressUtil.parseBit(p.getAddress()));
        } else if (p.getDataType() == EDataType.STRING) {
            RequestItem requestItem = AddressUtil.parseByte(p.getAddress(), 1 + p.getCount() * p.getDataType().getByteLength());
            int offset = this.s7PLC.getPlcType() == EPlcType.S200_SMART ? 0 : 1;
            requestItem.setByteAddress(requestItem.getByteAddress() + offset);
            s7ParseData.setRequestItem(requestItem);
        } else {
            s7ParseData.setRequestItem(AddressUtil.parseByte(p.getAddress(), p.getCount() * p.getDataType().getByteLength()));
        }
        return s7ParseData;
    }

    @Override
    public <T> T read(Class<T> targetClass) {
        List<S7ParseData> s7ParseDataList = this.parseBean(targetClass);
        this.readDataByCondition(s7ParseDataList);
        return this.fillData(targetClass, s7ParseDataList);
    }

    public List<S7Parameter> read(List<S7Parameter> parameters) {
        List<S7ParseData> s7ParseDataList = this.parseBean(parameters);
        this.readDataByCondition(s7ParseDataList);
        this.fillData(parameters, s7ParseDataList);
        return parameters;
    }

    private <T> T fillData(Class<T> targetClass, List<S7ParseData> s7ParseDataList) {
        try {
            T result = targetClass.newInstance();
            for (S7ParseData item : s7ParseDataList) {
                this.fillField(result, item);
            }
            return result;
        }
        catch (Exception e) {
            throw new S7CommException("Serialization fetch data error:" + e.getMessage(), e);
        }
    }

    private void fillData(List<S7Parameter> parameters, List<S7ParseData> s7ParseDataList) {
        try {
            for (int i = 0; i < s7ParseDataList.size(); ++i) {
                S7ParseData item = s7ParseDataList.get(i);
                S7Parameter parameter = parameters.get(i);
                this.fillField(parameter, item);
            }
        }
        catch (Exception e) {
            throw new S7CommException("Serialization fetch data error:" + e.getMessage(), e);
        }
    }

    private <T> void fillField(T result, S7ParseData item) throws IllegalAccessException {
        ByteReadBuff buff = new ByteReadBuff(item.getDataItem().getData());
        item.getField().setAccessible(true);
        switch (item.getDataType()) {
            case BOOL: {
                item.getField().set(result, buff.getBoolean(0));
                break;
            }
            case BYTE: {
                item.getField().set(result, buff.getBytes(item.getCount()));
                break;
            }
            case UINT16: {
                item.getField().set(result, buff.getUInt16());
                break;
            }
            case INT16: {
                item.getField().set(result, buff.getInt16());
                break;
            }
            case TIME: 
            case UINT32: {
                item.getField().set(result, buff.getUInt32());
                break;
            }
            case INT32: {
                item.getField().set(result, buff.getInt32());
                break;
            }
            case INT64: {
                item.getField().set(result, buff.getInt64());
                break;
            }
            case FLOAT32: {
                item.getField().set(result, Float.valueOf(buff.getFloat32()));
                break;
            }
            case FLOAT64: {
                item.getField().set(result, buff.getFloat64());
                break;
            }
            case STRING: {
                int length = buff.getByteToInt(0);
                item.getField().set(result, buff.getString(1, Math.min(length, item.getCount()), this.charsets));
                break;
            }
            case DATE: {
                LocalDate date = this.s7PLC.byteArrayToDate(buff.getBytes(2));
                item.getField().set(result, date);
                break;
            }
            case TIME_OF_DAY: {
                LocalTime time = this.s7PLC.byteArrayToTimeOfDay(buff.getBytes(4));
                item.getField().set(result, time);
                break;
            }
            case DTL: {
                LocalDateTime dateTime = this.s7PLC.byteArrayToDTL(buff.getBytes(12));
                item.getField().set(result, dateTime);
                break;
            }
            default: {
                throw new S7CommException("Data type can not be recognized");
            }
        }
    }

    @Override
    public <T> void write(T targetBean) {
        List<S7ParseData> s7ParseDataList = this.parseBean(targetBean.getClass());
        if (s7ParseDataList.isEmpty()) {
            throw new S7CommException("The number of parsed annotation data is empty, and the data cannot be read");
        }
        s7ParseDataList = this.extractData(targetBean, s7ParseDataList);
        List<RequestItem> requestItems = s7ParseDataList.stream().map(S7ParseData::getRequestItem).collect(Collectors.toList());
        List<DataItem> dataItems = s7ParseDataList.stream().map(S7ParseData::getDataItem).collect(Collectors.toList());
        this.s7PLC.writeS7Data(requestItems, dataItems);
    }

    public void write(List<S7Parameter> parameters) {
        List<S7ParseData> s7ParseDataList = this.parseBean(parameters);
        if (s7ParseDataList.size() != parameters.size()) {
            throw new S7CommException("The number of parsed data is inconsistent with the number of incoming data");
        }
        s7ParseDataList = this.extractData(parameters, s7ParseDataList);
        List<RequestItem> requestItems = s7ParseDataList.stream().map(S7ParseData::getRequestItem).collect(Collectors.toList());
        List<DataItem> dataItems = s7ParseDataList.stream().map(S7ParseData::getDataItem).collect(Collectors.toList());
        this.s7PLC.writeS7Data(requestItems, dataItems);
    }

    private <T> List<S7ParseData> extractData(T targetBean, List<S7ParseData> s7ParseDataList) {
        try {
            for (S7ParseData item : s7ParseDataList) {
                item.getField().setAccessible(true);
                Object data = item.getField().get(targetBean);
                if (data == null) continue;
                this.extractField(item, data);
            }
            return s7ParseDataList.stream().filter(x -> x.getDataItem() != null).collect(Collectors.toList());
        }
        catch (Exception e) {
            throw new S7CommException("Serialized fill byte data error:" + e.getMessage(), e);
        }
    }

    private List<S7ParseData> extractData(List<S7Parameter> parameters, List<S7ParseData> s7ParseDataList) {
        try {
            for (int i = 0; i < s7ParseDataList.size(); ++i) {
                S7ParseData item = s7ParseDataList.get(i);
                S7Parameter parameter = parameters.get(i);
                if (parameter.getValue() == null) {
                    throw new S7CommException("The value of the parameter cannot be null");
                }
                this.extractField(item, parameter.getValue());
            }
            return s7ParseDataList.stream().filter(x -> x.getDataItem() != null).collect(Collectors.toList());
        }
        catch (Exception e) {
            throw new S7CommException("Serialized fill byte data error:" + e.getMessage(), e);
        }
    }

    private void extractField(S7ParseData item, Object data) {
        switch (item.getDataType()) {
            case BOOL: {
                item.setDataItem(DataItem.createReqByBoolean((Boolean)data));
                break;
            }
            case BYTE: {
                item.setDataItem(DataItem.createReqByByte(ByteReadBuff.newInstance((byte[])data).getBytes(item.getCount())));
                break;
            }
            case UINT16: {
                item.setDataItem(DataItem.createReqByByte(ByteWriteBuff.newInstance(2).putShort((Integer)data).getData()));
                break;
            }
            case INT16: {
                item.setDataItem(DataItem.createReqByByte(ByteWriteBuff.newInstance(2).putShort((Short)data).getData()));
                break;
            }
            case TIME: 
            case UINT32: {
                item.setDataItem(DataItem.createReqByByte(ByteWriteBuff.newInstance(4).putInteger((Long)data).getData()));
                break;
            }
            case INT32: {
                item.setDataItem(DataItem.createReqByByte(ByteWriteBuff.newInstance(4).putInteger((Integer)data).getData()));
                break;
            }
            case INT64: {
                item.setDataItem(DataItem.createReqByByte(ByteWriteBuff.newInstance(8).putLong((Long)data).getData()));
                break;
            }
            case FLOAT32: {
                item.setDataItem(DataItem.createReqByByte(ByteWriteBuff.newInstance(4).putFloat(((Float)data).floatValue()).getData()));
                break;
            }
            case FLOAT64: {
                item.setDataItem(DataItem.createReqByByte(ByteWriteBuff.newInstance(8).putDouble((Double)data).getData()));
                break;
            }
            case STRING: {
                byte[] bytes = ((String)data).getBytes(this.charsets);
                int actualLength = Math.min(bytes.length, item.getCount());
                byte[] targetBytes = new byte[1 + actualLength];
                targetBytes[0] = (byte)actualLength;
                System.arraycopy(bytes, 0, targetBytes, 1, actualLength);
                item.setDataItem(DataItem.createReqByByte(targetBytes));
                item.getRequestItem().setCount(targetBytes.length);
                break;
            }
            case DATE: {
                byte[] dateBytes = this.s7PLC.dateToByteArray((LocalDate)data);
                item.setDataItem(DataItem.createReqByByte(dateBytes));
                break;
            }
            case TIME_OF_DAY: {
                byte[] timeBytes = this.s7PLC.timeOfDayToByteArray((LocalTime)data);
                item.setDataItem(DataItem.createReqByByte(timeBytes));
                break;
            }
            case DTL: {
                byte[] dateTimeData = this.s7PLC.dtlToByteArray((LocalDateTime)data);
                item.setDataItem(DataItem.createReqByByte(dateTimeData));
                break;
            }
            default: {
                throw new S7CommException("Data type can not be recognized");
            }
        }
    }

    public S7PLC getS7PLC() {
        return this.s7PLC;
    }

    public Charset getCharsets() {
        return this.charsets;
    }

    public void setCharsets(Charset charsets) {
        this.charsets = charsets;
    }

    public boolean equals(Object o) {
        if (o == this) {
            return true;
        }
        if (!(o instanceof S7Serializer)) {
            return false;
        }
        S7Serializer other = (S7Serializer)o;
        if (!other.canEqual(this)) {
            return false;
        }
        S7PLC this$s7PLC = this.getS7PLC();
        S7PLC other$s7PLC = other.getS7PLC();
        if (this$s7PLC == null ? other$s7PLC != null : !((Object)this$s7PLC).equals(other$s7PLC)) {
            return false;
        }
        Charset this$charsets = this.getCharsets();
        Charset other$charsets = other.getCharsets();
        return !(this$charsets == null ? other$charsets != null : !((Object)this$charsets).equals(other$charsets));
    }

    protected boolean canEqual(Object other) {
        return other instanceof S7Serializer;
    }

    public int hashCode() {
        int PRIME = 59;
        int result = 1;
        S7PLC $s7PLC = this.getS7PLC();
        result = result * 59 + ($s7PLC == null ? 43 : ((Object)$s7PLC).hashCode());
        Charset $charsets = this.getCharsets();
        result = result * 59 + ($charsets == null ? 43 : ((Object)$charsets).hashCode());
        return result;
    }

    public String toString() {
        return "S7Serializer(s7PLC=" + this.getS7PLC() + ", charsets=" + this.getCharsets() + ")";
    }
}

