/*
 * Copyright © 2023 广州宏天软件股份有限公司 hotent
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.hotent.tenent.aipsdk.model;

/**
 * @author: 黎扬贵
 * @date: 2025/2/14 16:05
 * @version: 1.0
 */

import java.io.Serializable;

/**
 * 通用返回结果
 *
 * @author 欧阳高龙
 * @company 广州宏天软件股份有限公司
 * @since 2023/4/4 10:45
 */
public class SdkResult<E> implements Serializable {
    private static final long serialVersionUID = 1L;
    /**
     * 状态 true：操作成功 false：操作失败
     */
    Boolean state = true;
    /**
     * 提示信息
     */
    String message;

    /**
     * 返回的数据
     */
    E value;

    /**
     * 标准的http状态码：200, 400, 401, 500等
     */
    private Integer code = 200;

    /**
     * 错误代码
     */
    private String errorCode;

    /**
     * 日志ID
     */
    private String logId;

    /**
     * 分布式链路ID
     */
    private String traceId;

    /**
     * 耗时（ms）
     */
    private Long spendTimeMill;

    /**
     * 构造成功的返回结果
     * <p>
     * 例如：SdkResult.ok(); 或者 SdkResult.&lt;String>ok();<br />
     * 结果集中各属性默认值如下：<br />
     * state: true<br />
     * message: "操作成功"<br />
     * code: 200
     * </p>
     *
     * @param <E>
     * @return
     */
    public static <E> SdkResult<E> ok() {
        return new SdkResult<>(true, "操作成功");
    }

    /**
     * 成功返回数据接口
     *
     * @param e 响应数据
     * @param <E>
     * @return
     */
    public static <E> SdkResult<E> success(E e) {
        SdkResult<E> cr = new SdkResult<E>();
        cr.setState(true);
        cr.setValue(e);
        cr.setCode(200);
        return cr;
    }

    /**
     * 失败
     *
     * @param message 异常信息
     * @return
     */
    public static <E> SdkResult<E> error(String message) {
        SdkResult<E> cr = new SdkResult<>(false, message, null);
        return cr;
    }

    /**
     * 构造成功的返回结果
     * <p>
     * 例如：SdkResult.error(); 或者 SdkResult.&lt;String>error();<br />
     * 结果集中各属性默认值如下：<br />
     * state: false<br />
     * message: "系统异常"<br />
     * code: 500
     * </p>
     *
     * @param <E>
     * @return
     */
    public static <E> SdkResult<E> error() {
        return new SdkResult<>(false, "系统异常");
    }

    /**
     * 设置message并返回结果
     *
     * @param message
     * @return
     */
    public SdkResult<E> message(String message) {
        this.setMessage(message);
        return this;
    }

    /**
     * 设置value并返回结果
     *
     * @param val
     * @return
     */
    public SdkResult<E> value(E val) {
        this.setValue(val);
        return this;
    }

    /**
     * 设置logId并返回结果
     *
     * @param logId
     * @return
     */
    public SdkResult<E> log(String logId) {
        this.setLogId(logId);
        return this;
    }

    public SdkResult() {}

    /**
     * 返回成功及成功的提示信息
     *
     * @param message
     */
    public SdkResult(String message) {
        this(true, message, null);
    }

    /**
     * 返回成功/失败，及对应的成功/失败提示信息
     *
     * @param state
     * @param message
     */
    public SdkResult(boolean state, String message) {
        this(state, message, null);
    }

    /**
     * 返回成功/失败，及对应的成功/失败提示信息，还有返回对应的数据
     *
     * @param state
     * @param message
     * @param value
     */
    public SdkResult(boolean state, String message, E value) {
        this.state = state;
        this.message = message;
        this.value = value;
    }

    public Boolean getState() {
        return state;
    }

    public void setState(Boolean state) {
        this.state = state;
    }

    public String getMessage() {
        return message;
    }

    public void setMessage(String message) {
        this.message = message;
    }

    public E getValue() {
        return value;
    }

    public void setValue(E value) {
        this.value = value;
    }

    public String getLogId() {
        return logId;
    }

    public void setLogId(String logId) {
        this.logId = logId;
    }

    public Integer getCode() {
        return code;
    }

    public void setCode(Integer code) {
        this.code = code;
    }

    public String getErrorCode() {
        return errorCode;
    }

    public void setErrorCode(String errorCode) {
        this.errorCode = errorCode;
    }

    public String getTraceId() {
        return this.logId;
    }

    public Long getSpendTimeMill() {
        return spendTimeMill;
    }

    public void setSpendTimeMill(Long spendTimeMill) {
        this.spendTimeMill = spendTimeMill;
    }
}
