/*
 * Copyright © 2023 广州宏天软件股份有限公司 hotent
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.hotent.tenent.aipsdk.service;

import com.fasterxml.jackson.databind.node.ObjectNode;
import com.github.benmanes.caffeine.cache.Cache;
import com.hotent.tenent.aipsdk.client.BaiduAipSdkFeignClient;
import com.hotent.tenent.aipsdk.model.TokenResponse;
import com.hotent.tenent.aipsdk.properties.BaiduAipSdkProperties;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.time.Instant;

/**
 * Token 服务类，负责管理 Token 的获取和缓存。
 *
 * @author: 黎扬贵
 * @date: 2025/2/14 14:12
 * @version: 1.0
 */
@Service
public class TokenService {
    @Autowired
    private BaiduAipSdkProperties baiduAipSdkProperties;
    @Autowired
    private BaiduAipSdkFeignClient baiduAipSdkFeignClient;
    @Autowired
    private Cache<String, TokenResponse> tokenCache;

    /**
     * 获取有效的 Token。 首先检查缓存中是否存在有效的 Token，如果存在则直接返回； 如果不存在或已过期，则通过 RESTful 接口获取新的 Token 并放入缓存。
     *
     * @return 有效的 Token
     */
    public synchronized String getAccessToken() {
        String cacheKey = baiduAipSdkProperties.getAppKey() + ":" + baiduAipSdkProperties.getSecretKey();
        TokenResponse tokenResponse = tokenCache.getIfPresent(cacheKey);
        if (tokenResponse != null && isTokenValid(tokenResponse)) {
            return tokenResponse.getAccessToken();
        }
        ObjectNode accessToken = baiduAipSdkFeignClient.getAccessToken(baiduAipSdkProperties.getAppKey(), baiduAipSdkProperties.getSecretKey());
        if (accessToken == null) {
            throw new RuntimeException("获取 Token 失败");
        }
        if (accessToken.has("error")) {
            throw new RuntimeException(accessToken.get("error_description").asText());
        }
        String accessTokenStr = accessToken.get("access_token").asText();
        int expiresIn = accessToken.get("expires_in").asInt();
        tokenResponse = new TokenResponse();
        tokenResponse.setAccessToken(accessTokenStr);
        // 让token提前5秒过期，重新获取token
        tokenResponse.setExpireIn(Instant.now().getEpochSecond() + expiresIn - 5);
        tokenCache.put(cacheKey, tokenResponse);
        return tokenResponse.getAccessToken();
    }

    /**
     * 检查 Token 是否有效。
     *
     * @param tokenResponse Token 响应对象
     * @return 如果 Token 未过期则返回 true，否则返回 false
     */
    private boolean isTokenValid(TokenResponse tokenResponse) {
        return tokenResponse.getExpireIn() > Instant.now().getEpochSecond();
    }
}