/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.io.network.partition;

import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.Arrays;
import javax.annotation.concurrent.NotThreadSafe;
import org.apache.flink.runtime.io.network.buffer.Buffer;
import org.apache.flink.runtime.io.network.partition.BufferReaderWriterUtil;
import org.apache.flink.runtime.io.network.partition.PartitionedFile;
import org.apache.flink.util.ExceptionUtils;
import org.apache.flink.util.IOUtils;
import org.apache.flink.util.Preconditions;

@NotThreadSafe
public class PartitionedFileWriter
implements AutoCloseable {
    private static final int MIN_INDEX_BUFFER_SIZE = 600;
    private final ByteBuffer[] header = BufferReaderWriterUtil.allocatedWriteBufferArray();
    private final int numSubpartitions;
    private final FileChannel dataFileChannel;
    private final FileChannel indexFileChannel;
    private final Path dataFilePath;
    private final Path indexFilePath;
    private final long[] subpartitionOffsets;
    private final int[] subpartitionBuffers;
    private ByteBuffer writeDataCache;
    private final int maxIndexBufferSize;
    private ByteBuffer indexBuffer;
    private boolean allIndexEntriesCached = true;
    private long totalBytesWritten;
    private int numRegions;
    private int currentSubpartition = -1;
    private boolean isFinished;
    private boolean isClosed;

    public PartitionedFileWriter(int numSubpartitions, int maxIndexBufferSize, String basePath) throws IOException {
        Preconditions.checkArgument((numSubpartitions > 0 ? 1 : 0) != 0, (Object)"Illegal number of subpartitions.");
        Preconditions.checkArgument((maxIndexBufferSize > 0 ? 1 : 0) != 0, (Object)"Illegal maximum index cache size.");
        Preconditions.checkArgument((basePath != null ? 1 : 0) != 0, (Object)"Base path must not be null.");
        this.numSubpartitions = numSubpartitions;
        this.maxIndexBufferSize = this.alignMaxIndexBufferSize(maxIndexBufferSize);
        this.subpartitionOffsets = new long[numSubpartitions];
        this.subpartitionBuffers = new int[numSubpartitions];
        this.dataFilePath = new File(basePath + ".shuffle.data").toPath();
        this.indexFilePath = new File(basePath + ".shuffle.index").toPath();
        this.indexBuffer = ByteBuffer.allocateDirect(600);
        BufferReaderWriterUtil.configureByteBuffer(this.indexBuffer);
        this.writeDataCache = ByteBuffer.allocateDirect(0x400000);
        BufferReaderWriterUtil.configureByteBuffer(this.writeDataCache);
        this.dataFileChannel = this.openFileChannel(this.dataFilePath);
        try {
            this.indexFileChannel = this.openFileChannel(this.indexFilePath);
        }
        catch (Throwable throwable) {
            IOUtils.closeQuietly((AutoCloseable)this.dataFileChannel);
            IOUtils.deleteFileQuietly((Path)this.dataFilePath);
            throw throwable;
        }
    }

    private FileChannel openFileChannel(Path path) throws IOException {
        return FileChannel.open(path, StandardOpenOption.CREATE_NEW, StandardOpenOption.WRITE);
    }

    private int alignMaxIndexBufferSize(int maxIndexBufferSize) {
        return maxIndexBufferSize / 12 * 12;
    }

    public void startNewRegion() throws IOException {
        Preconditions.checkState((!this.isFinished ? 1 : 0) != 0, (Object)"File writer is already finished.");
        Preconditions.checkState((!this.isClosed ? 1 : 0) != 0, (Object)"File writer is already closed.");
        this.writeRegionIndex();
    }

    private void writeIndexEntry(long subpartitionOffset, int numBuffers) throws IOException {
        if (!this.indexBuffer.hasRemaining() && !this.extendIndexBufferIfPossible()) {
            this.flushIndexBuffer();
            this.indexBuffer.clear();
            this.allIndexEntriesCached = false;
        }
        this.indexBuffer.putLong(subpartitionOffset);
        this.indexBuffer.putInt(numBuffers);
    }

    private boolean extendIndexBufferIfPossible() {
        if (this.indexBuffer.capacity() >= this.maxIndexBufferSize) {
            return false;
        }
        int newIndexBufferSize = Math.min(this.maxIndexBufferSize, 2 * this.indexBuffer.capacity());
        ByteBuffer newIndexBuffer = ByteBuffer.allocateDirect(newIndexBufferSize);
        this.indexBuffer.flip();
        newIndexBuffer.put(this.indexBuffer);
        BufferReaderWriterUtil.configureByteBuffer(newIndexBuffer);
        this.indexBuffer = newIndexBuffer;
        return true;
    }

    private void writeRegionIndex() throws IOException {
        if (Arrays.stream(this.subpartitionBuffers).sum() > 0) {
            for (int channel = 0; channel < this.numSubpartitions; ++channel) {
                this.writeIndexEntry(this.subpartitionOffsets[channel], this.subpartitionBuffers[channel]);
            }
            this.currentSubpartition = -1;
            ++this.numRegions;
            Arrays.fill(this.subpartitionBuffers, 0);
        }
    }

    private void flushIndexBuffer() throws IOException {
        this.indexBuffer.flip();
        if (this.indexBuffer.limit() > 0) {
            BufferReaderWriterUtil.writeBuffer(this.indexFileChannel, this.indexBuffer);
        }
    }

    public void writeBuffer(Buffer target, int targetSubpartition) throws IOException {
        Preconditions.checkState((!this.isFinished ? 1 : 0) != 0, (Object)"File writer is already finished.");
        Preconditions.checkState((!this.isClosed ? 1 : 0) != 0, (Object)"File writer is already closed.");
        if (targetSubpartition != this.currentSubpartition) {
            Preconditions.checkState((this.subpartitionBuffers[targetSubpartition] == 0 ? 1 : 0) != 0, (Object)"Must write data of the same channel together.");
            this.subpartitionOffsets[targetSubpartition] = this.totalBytesWritten;
            this.currentSubpartition = targetSubpartition;
        }
        this.totalBytesWritten += BufferReaderWriterUtil.writeToByteChannel(this.dataFileChannel, target, this.writeDataCache, this.header);
        int n = targetSubpartition;
        this.subpartitionBuffers[n] = this.subpartitionBuffers[n] + 1;
    }

    public PartitionedFile finish() throws IOException {
        Preconditions.checkState((!this.isFinished ? 1 : 0) != 0, (Object)"File writer is already finished.");
        Preconditions.checkState((!this.isClosed ? 1 : 0) != 0, (Object)"File writer is already closed.");
        this.isFinished = true;
        this.writeDataCache.flip();
        if (this.writeDataCache.hasRemaining()) {
            BufferReaderWriterUtil.writeBuffer(this.dataFileChannel, this.writeDataCache);
        }
        this.writeDataCache = null;
        this.writeRegionIndex();
        this.flushIndexBuffer();
        this.indexBuffer.rewind();
        this.close();
        ByteBuffer indexEntryCache = null;
        if (this.allIndexEntriesCached) {
            indexEntryCache = this.indexBuffer;
        }
        this.indexBuffer = null;
        return new PartitionedFile(this.numRegions, this.numSubpartitions, this.dataFilePath, this.indexFilePath, indexEntryCache);
    }

    public void releaseQuietly() {
        IOUtils.closeQuietly((AutoCloseable)this);
        IOUtils.deleteFileQuietly((Path)this.dataFilePath);
        IOUtils.deleteFileQuietly((Path)this.indexFilePath);
    }

    @Override
    public void close() throws IOException {
        if (this.isClosed) {
            return;
        }
        this.isClosed = true;
        IOException exception = null;
        try {
            this.dataFileChannel.close();
        }
        catch (IOException ioException) {
            exception = ioException;
        }
        try {
            this.indexFileChannel.close();
        }
        catch (IOException ioException) {
            exception = (IOException)ExceptionUtils.firstOrSuppressed((Throwable)ioException, (Throwable)exception);
        }
        if (exception != null) {
            throw exception;
        }
    }
}

