/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.replication.regionserver;

import java.io.EOFException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.PriorityBlockingQueue;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.Cell;
import org.apache.hadoop.hbase.CellUtil;
import org.apache.hadoop.hbase.replication.WALEntryFilter;
import org.apache.hadoop.hbase.replication.regionserver.ReplicationSource;
import org.apache.hadoop.hbase.replication.regionserver.WALEntryStream;
import org.apache.hadoop.hbase.shaded.protobuf.generated.WALProtos;
import org.apache.hadoop.hbase.util.Pair;
import org.apache.hadoop.hbase.util.Threads;
import org.apache.hadoop.hbase.wal.WAL;
import org.apache.hadoop.hbase.wal.WALEdit;
import org.apache.yetus.audience.InterfaceAudience;
import org.apache.yetus.audience.InterfaceStability;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
@InterfaceStability.Evolving
public class ReplicationSourceWALReader
extends Thread {
    private static final Logger LOG = LoggerFactory.getLogger(ReplicationSourceWALReader.class);
    private final PriorityBlockingQueue<Path> logQueue;
    private final FileSystem fs;
    private final Configuration conf;
    private final WALEntryFilter filter;
    private final ReplicationSource source;
    protected final BlockingQueue<WALEntryBatch> entryBatchQueue;
    private final long replicationBatchSizeCapacity;
    protected final int replicationBatchCountCapacity;
    private long currentPosition;
    private final long sleepForRetries;
    private final int maxRetriesMultiplier;
    private final boolean eofAutoRecovery;
    private boolean isReaderRunning = true;
    private AtomicLong totalBufferUsed;
    private long totalBufferQuota;

    public ReplicationSourceWALReader(FileSystem fs, Configuration conf, PriorityBlockingQueue<Path> logQueue, long startPosition, WALEntryFilter filter, ReplicationSource source) {
        this.logQueue = logQueue;
        this.currentPosition = startPosition;
        this.fs = fs;
        this.conf = conf;
        this.filter = filter;
        this.source = source;
        this.replicationBatchSizeCapacity = this.conf.getLong("replication.source.size.capacity", 0x4000000L);
        this.replicationBatchCountCapacity = this.conf.getInt("replication.source.nb.capacity", 25000);
        int batchCount = conf.getInt("replication.source.nb.batches", 1);
        this.totalBufferUsed = source.getSourceManager().getTotalBufferUsed();
        this.totalBufferQuota = conf.getLong("replication.total.buffer.quota", 0x10000000L);
        this.sleepForRetries = this.conf.getLong("replication.source.sleepforretries", 1000L);
        this.maxRetriesMultiplier = this.conf.getInt("replication.source.maxretriesmultiplier", 300);
        this.eofAutoRecovery = conf.getBoolean("replication.source.eof.autorecovery", false);
        this.entryBatchQueue = new LinkedBlockingQueue<WALEntryBatch>(batchCount);
        LOG.info("peerClusterZnode=" + source.getPeerClusterZnode() + ", ReplicationSourceWALReaderThread : " + source.getPeerId() + " inited, replicationBatchSizeCapacity=" + this.replicationBatchSizeCapacity + ", replicationBatchCountCapacity=" + this.replicationBatchCountCapacity + ", replicationBatchQueueCapacity=" + batchCount);
    }

    @Override
    public void run() {
        int sleepMultiplier = 1;
        while (this.isReaderRunning()) {
            try {
                WALEntryStream entryStream = new WALEntryStream(this.logQueue, this.fs, this.conf, this.currentPosition, this.source.getWALFileLengthProvider(), this.source.getServerWALsBelongTo(), this.source.getSourceMetrics());
                Throwable throwable = null;
                try {
                    while (this.isReaderRunning()) {
                        if (!this.checkQuota()) continue;
                        WALEntryBatch batch = this.readWALEntries(entryStream);
                        if (batch != null && batch.getNbEntries() > 0) {
                            if (LOG.isTraceEnabled()) {
                                LOG.trace(String.format("Read %s WAL entries eligible for replication", batch.getNbEntries()));
                            }
                            this.entryBatchQueue.put(batch);
                            sleepMultiplier = 1;
                        } else {
                            this.handleEmptyWALEntryBatch(batch, entryStream.getCurrentPath());
                        }
                        this.currentPosition = entryStream.getPosition();
                        entryStream.reset();
                    }
                }
                catch (Throwable throwable2) {
                    throwable = throwable2;
                    throw throwable2;
                }
                finally {
                    if (entryStream == null) continue;
                    if (throwable != null) {
                        try {
                            entryStream.close();
                        }
                        catch (Throwable throwable3) {
                            throwable.addSuppressed(throwable3);
                        }
                        continue;
                    }
                    entryStream.close();
                }
            }
            catch (IOException e) {
                if (sleepMultiplier < this.maxRetriesMultiplier) {
                    LOG.debug("Failed to read stream of replication entries: " + e);
                    ++sleepMultiplier;
                } else {
                    LOG.error("Failed to read stream of replication entries", (Throwable)e);
                    this.handleEofException(e);
                }
                Threads.sleep((long)(this.sleepForRetries * (long)sleepMultiplier));
            }
            catch (InterruptedException e) {
                LOG.trace("Interrupted while sleeping between WAL reads");
                Thread.currentThread().interrupt();
            }
        }
    }

    private WALEntryBatch readWALEntries(WALEntryStream entryStream) throws IOException {
        WALEntryBatch batch = null;
        while (entryStream.hasNext()) {
            WALEdit edit;
            if (batch == null) {
                batch = new WALEntryBatch(this.replicationBatchCountCapacity, entryStream.getCurrentPath());
            }
            WAL.Entry entry = entryStream.next();
            if ((entry = this.filterEntry(entry)) == null || (edit = entry.getEdit()) == null || edit.isEmpty()) continue;
            long entrySize = this.getEntrySize(entry);
            batch.addEntry(entry);
            this.updateBatchStats(batch, entry, entryStream.getPosition(), entrySize);
            boolean totalBufferTooLarge = this.acquireBufferQuota(entrySize);
            if (!totalBufferTooLarge && batch.getHeapSize() < this.replicationBatchSizeCapacity && batch.getNbEntries() < this.replicationBatchCountCapacity) continue;
            break;
        }
        return batch;
    }

    protected void handleEmptyWALEntryBatch(WALEntryBatch batch, Path currentPath) throws InterruptedException {
        LOG.trace("Didn't read any new entries from WAL");
        Thread.sleep(this.sleepForRetries);
    }

    private void handleEofException(IOException e) {
        if ((e instanceof EOFException || e.getCause() instanceof EOFException) && this.logQueue.size() > 1 && this.eofAutoRecovery) {
            try {
                if (this.fs.getFileStatus(this.logQueue.peek()).getLen() == 0L) {
                    LOG.warn("Forcing removal of 0 length log in queue: " + this.logQueue.peek());
                    this.logQueue.remove();
                    this.currentPosition = 0L;
                }
            }
            catch (IOException ioe) {
                LOG.warn("Couldn't get file length information about log " + this.logQueue.peek());
            }
        }
    }

    public Path getCurrentPath() {
        WALEntryBatch batchQueueHead = (WALEntryBatch)this.entryBatchQueue.peek();
        if (batchQueueHead != null) {
            return batchQueueHead.lastWalPath;
        }
        return this.logQueue.peek();
    }

    private boolean checkQuota() {
        if (this.totalBufferUsed.get() > this.totalBufferQuota) {
            Threads.sleep((long)this.sleepForRetries);
            return false;
        }
        return true;
    }

    private WAL.Entry filterEntry(WAL.Entry entry) {
        WAL.Entry filtered = this.filter.filter(entry);
        if (entry != null && filtered == null) {
            this.source.getSourceMetrics().incrLogEditsFiltered();
        }
        return filtered;
    }

    public WALEntryBatch take() throws InterruptedException {
        return this.entryBatchQueue.take();
    }

    private long getEntrySize(WAL.Entry entry) {
        WALEdit edit = entry.getEdit();
        return edit.heapSize() + (long)this.calculateTotalSizeOfStoreFiles(edit);
    }

    private void updateBatchStats(WALEntryBatch batch, WAL.Entry entry, long entryPosition, long entrySize) {
        WALEdit edit = entry.getEdit();
        if (edit != null && !edit.isEmpty()) {
            batch.incrementHeapSize(entrySize);
            Pair<Integer, Integer> nbRowsAndHFiles = this.countDistinctRowKeysAndHFiles(edit);
            batch.incrementNbRowKeys((Integer)nbRowsAndHFiles.getFirst());
            batch.incrementNbHFiles((Integer)nbRowsAndHFiles.getSecond());
        }
        batch.lastWalPosition = entryPosition;
    }

    private Pair<Integer, Integer> countDistinctRowKeysAndHFiles(WALEdit edit) {
        ArrayList<Cell> cells = edit.getCells();
        int distinctRowKeys = 1;
        int totalHFileEntries = 0;
        Cell lastCell = (Cell)cells.get(0);
        int totalCells = edit.size();
        for (int i = 0; i < totalCells; ++i) {
            if (CellUtil.matchingQualifier((Cell)((Cell)cells.get(i)), (byte[])WALEdit.BULK_LOAD)) {
                try {
                    WALProtos.BulkLoadDescriptor bld = WALEdit.getBulkLoadDescriptor((Cell)cells.get(i));
                    List stores = bld.getStoresList();
                    int totalStores = stores.size();
                    for (int j = 0; j < totalStores; ++j) {
                        totalHFileEntries += ((WALProtos.StoreDescriptor)stores.get(j)).getStoreFileList().size();
                    }
                }
                catch (IOException e) {
                    LOG.error("Failed to deserialize bulk load entry from wal edit. Then its hfiles count will not be added into metric.");
                }
            }
            if (!CellUtil.matchingRows((Cell)((Cell)cells.get(i)), (Cell)lastCell)) {
                ++distinctRowKeys;
            }
            lastCell = (Cell)cells.get(i);
        }
        Pair result = new Pair((Object)distinctRowKeys, (Object)totalHFileEntries);
        return result;
    }

    private int calculateTotalSizeOfStoreFiles(WALEdit edit) {
        ArrayList<Cell> cells = edit.getCells();
        int totalStoreFilesSize = 0;
        int totalCells = edit.size();
        for (int i = 0; i < totalCells; ++i) {
            if (!CellUtil.matchingQualifier((Cell)((Cell)cells.get(i)), (byte[])WALEdit.BULK_LOAD)) continue;
            try {
                WALProtos.BulkLoadDescriptor bld = WALEdit.getBulkLoadDescriptor((Cell)cells.get(i));
                List stores = bld.getStoresList();
                int totalStores = stores.size();
                for (int j = 0; j < totalStores; ++j) {
                    totalStoreFilesSize = (int)((long)totalStoreFilesSize + ((WALProtos.StoreDescriptor)stores.get(j)).getStoreFileSizeBytes());
                }
                continue;
            }
            catch (IOException e) {
                LOG.error("Failed to deserialize bulk load entry from wal edit. Size of HFiles part of cell will not be considered in replication request size calculation.", (Throwable)e);
            }
        }
        return totalStoreFilesSize;
    }

    private boolean acquireBufferQuota(long size) {
        return this.totalBufferUsed.addAndGet(size) >= this.totalBufferQuota;
    }

    public boolean isReaderRunning() {
        return this.isReaderRunning && !this.isInterrupted();
    }

    public void setReaderRunning(boolean readerRunning) {
        this.isReaderRunning = readerRunning;
    }

    static class WALEntryBatch {
        private List<WAL.Entry> walEntries;
        private Path lastWalPath;
        private long lastWalPosition = 0L;
        private int nbRowKeys = 0;
        private int nbHFiles = 0;
        private long heapSize = 0L;

        WALEntryBatch(int maxNbEntries, Path lastWalPath) {
            this.walEntries = new ArrayList<WAL.Entry>(maxNbEntries);
            this.lastWalPath = lastWalPath;
        }

        public void addEntry(WAL.Entry entry) {
            this.walEntries.add(entry);
        }

        public List<WAL.Entry> getWalEntries() {
            return this.walEntries;
        }

        public Path getLastWalPath() {
            return this.lastWalPath;
        }

        public long getLastWalPosition() {
            return this.lastWalPosition;
        }

        public int getNbEntries() {
            return this.walEntries.size();
        }

        public int getNbRowKeys() {
            return this.nbRowKeys;
        }

        public int getNbHFiles() {
            return this.nbHFiles;
        }

        public int getNbOperations() {
            return this.getNbRowKeys() + this.getNbHFiles();
        }

        public long getHeapSize() {
            return this.heapSize;
        }

        private void incrementNbRowKeys(int increment) {
            this.nbRowKeys += increment;
        }

        private void incrementNbHFiles(int increment) {
            this.nbHFiles += increment;
        }

        private void incrementHeapSize(long increment) {
            this.heapSize += increment;
        }
    }
}

