/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.io.hfile.bucket;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.Set;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.HBaseClassTestRule;
import org.apache.hadoop.hbase.HBaseConfiguration;
import org.apache.hadoop.hbase.HBaseTestingUtility;
import org.apache.hadoop.hbase.io.hfile.BlockCache;
import org.apache.hadoop.hbase.io.hfile.BlockCacheKey;
import org.apache.hadoop.hbase.io.hfile.BlockType;
import org.apache.hadoop.hbase.io.hfile.CacheTestUtils;
import org.apache.hadoop.hbase.io.hfile.Cacheable;
import org.apache.hadoop.hbase.io.hfile.HFileBlock;
import org.apache.hadoop.hbase.io.hfile.HFileContext;
import org.apache.hadoop.hbase.io.hfile.HFileContextBuilder;
import org.apache.hadoop.hbase.io.hfile.bucket.BucketAllocator;
import org.apache.hadoop.hbase.io.hfile.bucket.BucketAllocatorException;
import org.apache.hadoop.hbase.io.hfile.bucket.BucketCache;
import org.apache.hadoop.hbase.io.hfile.bucket.CacheFullException;
import org.apache.hadoop.hbase.testclassification.IOTests;
import org.apache.hadoop.hbase.testclassification.MediumTests;
import org.apache.hbase.thirdparty.com.google.common.collect.ImmutableMap;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

@RunWith(value=Parameterized.class)
@Category(value={IOTests.class, MediumTests.class})
public class TestBucketCache {
    @ClassRule
    public static final HBaseClassTestRule CLASS_RULE = HBaseClassTestRule.forClass(TestBucketCache.class);
    private static final Random RAND = new Random();
    @Parameterized.Parameter(value=0)
    public int constructedBlockSize;
    @Parameterized.Parameter(value=1)
    public int[] constructedBlockSizes;
    BucketCache cache;
    final int CACHE_SIZE = 1000000;
    final int NUM_BLOCKS = 100;
    final int BLOCK_SIZE = 10000;
    final int NUM_THREADS = 100;
    final int NUM_QUERIES = 10000;
    final long capacitySize = 0x2000000L;
    final int writeThreads = 3;
    final int writerQLen = 64;
    String ioEngineName = "offheap";
    String persistencePath = null;

    @Parameterized.Parameters(name="{index}: blockSize={0}, bucketSizes={1}")
    public static Iterable<Object[]> data() {
        return Arrays.asList({8192, null}, {16384, new int[]{3072, 5120, 9216, 17408, 29696, 33792, 66560, 99328, 132096}});
    }

    @Before
    public void setup() throws FileNotFoundException, IOException {
        this.cache = new MockedBucketCache(this.ioEngineName, 0x2000000L, this.constructedBlockSize, this.constructedBlockSizes, 3, 64, this.persistencePath);
    }

    @After
    public void tearDown() {
        this.cache.shutdown();
    }

    private static <T> T randFrom(List<T> a) {
        return a.get(RAND.nextInt(a.size()));
    }

    @Test
    public void testBucketAllocator() throws BucketAllocatorException {
        BucketAllocator mAllocator = this.cache.getAllocator();
        List<Integer> BLOCKSIZES = Arrays.asList(4096, 8192, 65536, 98304);
        boolean full = false;
        ArrayList<Long> allocations = new ArrayList<Long>();
        ArrayList<Integer> tmp = new ArrayList<Integer>(BLOCKSIZES);
        while (!full) {
            Integer blockSize = null;
            try {
                blockSize = TestBucketCache.randFrom(tmp);
                allocations.add(mAllocator.allocateBlock(blockSize.intValue()));
            }
            catch (CacheFullException cfe) {
                tmp.remove(blockSize);
                if (!tmp.isEmpty()) continue;
                full = true;
            }
        }
        for (Integer blockSize : BLOCKSIZES) {
            BucketAllocator.BucketSizeInfo bucketSizeInfo = mAllocator.roundUpToBucketSizeInfo(blockSize.intValue());
            BucketAllocator.IndexStatistics indexStatistics = bucketSizeInfo.statistics();
            Assert.assertEquals((String)("unexpected freeCount for " + bucketSizeInfo), (long)0L, (long)indexStatistics.freeCount());
        }
        Iterator<Integer> iterator = allocations.iterator();
        while (iterator.hasNext()) {
            long offset = (Long)((Object)iterator.next());
            Assert.assertEquals((long)mAllocator.sizeOfAllocation(offset), (long)mAllocator.freeBlock(offset));
        }
        Assert.assertEquals((long)0L, (long)mAllocator.getUsedSize());
    }

    @Test
    public void testCacheSimple() throws Exception {
        CacheTestUtils.testCacheSimple((BlockCache)this.cache, 10000, 10000);
    }

    @Test
    public void testCacheMultiThreadedSingleKey() throws Exception {
        CacheTestUtils.hammerSingleKey((BlockCache)this.cache, 10000, 200, 20000);
    }

    @Test
    public void testHeapSizeChanges() throws Exception {
        this.cache.stopWriterThreads();
        CacheTestUtils.testHeapSizeChanges((BlockCache)this.cache, 10000);
    }

    private void waitUntilFlushedToBucket(BucketCache cache, BlockCacheKey cacheKey) throws InterruptedException {
        while (!cache.backingMap.containsKey(cacheKey) || cache.ramCache.containsKey(cacheKey)) {
            Thread.sleep(100L);
        }
    }

    private void cacheAndWaitUntilFlushedToBucket(BucketCache cache, BlockCacheKey cacheKey, Cacheable block) throws InterruptedException {
        cache.cacheBlock(cacheKey, block);
        this.waitUntilFlushedToBucket(cache, cacheKey);
    }

    @Test
    public void testMemoryLeak() throws Exception {
        final BlockCacheKey cacheKey = new BlockCacheKey("dummy", 1L);
        this.cacheAndWaitUntilFlushedToBucket(this.cache, cacheKey, new CacheTestUtils.ByteArrayCacheable(new byte[10]));
        long lockId = ((BucketCache.BucketEntry)this.cache.backingMap.get(cacheKey)).offset();
        ReentrantReadWriteLock lock = this.cache.offsetLock.getLock((Object)lockId);
        lock.writeLock().lock();
        Thread evictThread = new Thread("evict-block"){

            @Override
            public void run() {
                TestBucketCache.this.cache.evictBlock(cacheKey);
            }
        };
        evictThread.start();
        this.cache.offsetLock.waitForWaiters((Object)lockId, 1);
        this.cache.blockEvicted(cacheKey, (BucketCache.BucketEntry)this.cache.backingMap.remove(cacheKey), true);
        this.cacheAndWaitUntilFlushedToBucket(this.cache, cacheKey, new CacheTestUtils.ByteArrayCacheable(new byte[10]));
        lock.writeLock().unlock();
        evictThread.join();
        Assert.assertEquals((long)1L, (long)this.cache.getBlockCount());
        Assert.assertTrue((this.cache.getCurrentSize() > 0L ? 1 : 0) != 0);
        Assert.assertTrue((String)"We should have a block!", (boolean)this.cache.iterator().hasNext());
    }

    @Test
    public void testRetrieveFromFile() throws Exception {
        CacheTestUtils.HFileBlockPair[] blocks;
        HBaseTestingUtility TEST_UTIL = new HBaseTestingUtility();
        Path testDir = TEST_UTIL.getDataTestDir();
        TEST_UTIL.getTestFileSystem().mkdirs(testDir);
        BucketCache bucketCache = new BucketCache("file:" + testDir + "/bucket.cache", 0x2000000L, this.constructedBlockSize, this.constructedBlockSizes, 3, 64, testDir + "/bucket.persistence");
        long usedSize = bucketCache.getAllocator().getUsedSize();
        Assert.assertTrue((usedSize == 0L ? 1 : 0) != 0);
        for (CacheTestUtils.HFileBlockPair block : blocks = CacheTestUtils.generateHFileBlocks(this.constructedBlockSize, 1)) {
            bucketCache.cacheBlock(block.getBlockName(), (Cacheable)block.getBlock());
        }
        for (CacheTestUtils.HFileBlockPair block : blocks) {
            this.cacheAndWaitUntilFlushedToBucket(bucketCache, block.getBlockName(), (Cacheable)block.getBlock());
        }
        usedSize = bucketCache.getAllocator().getUsedSize();
        Assert.assertTrue((usedSize != 0L ? 1 : 0) != 0);
        bucketCache.shutdown();
        bucketCache = new BucketCache("file:" + testDir + "/bucket.cache", 0x2000000L, this.constructedBlockSize, this.constructedBlockSizes, 3, 64, testDir + "/bucket.persistence");
        Assert.assertEquals((long)usedSize, (long)bucketCache.getAllocator().getUsedSize());
        bucketCache.shutdown();
        int[] smallBucketSizes = new int[]{3072, 5120};
        bucketCache = new BucketCache("file:" + testDir + "/bucket.cache", 0x2000000L, this.constructedBlockSize, smallBucketSizes, 3, 64, testDir + "/bucket.persistence");
        Assert.assertEquals((long)0L, (long)bucketCache.getAllocator().getUsedSize());
        Assert.assertEquals((long)0L, (long)bucketCache.backingMap.size());
        TEST_UTIL.cleanupTestDir();
    }

    @Test
    public void testBucketAllocatorLargeBuckets() throws BucketAllocatorException {
        long availableSpace = 0x500000000L;
        int[] bucketSizes = new int[]{1024, 0x100000, 0x40000000};
        BucketAllocator allocator = new BucketAllocator(availableSpace, bucketSizes);
        Assert.assertTrue((allocator.getBuckets().length > 0 ? 1 : 0) != 0);
    }

    @Test
    public void testGetPartitionSize() throws IOException {
        this.validateGetPartitionSize(this.cache, 0.25f, 0.85f);
        Configuration conf = HBaseConfiguration.create();
        conf.setFloat("hbase.bucketcache.minfactor", 0.5f);
        conf.setFloat("hbase.bucketcache.single.factor", 0.1f);
        conf.setFloat("hbase.bucketcache.multi.factor", 0.7f);
        conf.setFloat("hbase.bucketcache.memory.factor", 0.2f);
        BucketCache cache = new BucketCache(this.ioEngineName, 0x2000000L, this.constructedBlockSize, this.constructedBlockSizes, 3, 64, this.persistencePath, 100, conf);
        this.validateGetPartitionSize(cache, 0.1f, 0.5f);
        this.validateGetPartitionSize(cache, 0.7f, 0.5f);
        this.validateGetPartitionSize(cache, 0.2f, 0.5f);
    }

    @Test
    public void testValidBucketCacheConfigs() throws IOException {
        Configuration conf = HBaseConfiguration.create();
        conf.setFloat("hbase.bucketcache.acceptfactor", 0.9f);
        conf.setFloat("hbase.bucketcache.minfactor", 0.5f);
        conf.setFloat("hbase.bucketcache.extrafreefactor", 0.5f);
        conf.setFloat("hbase.bucketcache.single.factor", 0.1f);
        conf.setFloat("hbase.bucketcache.multi.factor", 0.7f);
        conf.setFloat("hbase.bucketcache.memory.factor", 0.2f);
        BucketCache cache = new BucketCache(this.ioEngineName, 0x2000000L, this.constructedBlockSize, this.constructedBlockSizes, 3, 64, this.persistencePath, 100, conf);
        Assert.assertEquals((String)"hbase.bucketcache.acceptfactor failed to propagate.", (float)0.9f, (float)cache.getAcceptableFactor(), (float)0.0f);
        Assert.assertEquals((String)"hbase.bucketcache.minfactor failed to propagate.", (float)0.5f, (float)cache.getMinFactor(), (float)0.0f);
        Assert.assertEquals((String)"hbase.bucketcache.extrafreefactor failed to propagate.", (float)0.5f, (float)cache.getExtraFreeFactor(), (float)0.0f);
        Assert.assertEquals((String)"hbase.bucketcache.single.factor failed to propagate.", (float)0.1f, (float)cache.getSingleFactor(), (float)0.0f);
        Assert.assertEquals((String)"hbase.bucketcache.multi.factor failed to propagate.", (float)0.7f, (float)cache.getMultiFactor(), (float)0.0f);
        Assert.assertEquals((String)"hbase.bucketcache.memory.factor failed to propagate.", (float)0.2f, (float)cache.getMemoryFactor(), (float)0.0f);
    }

    @Test
    public void testInvalidAcceptFactorConfig() throws IOException {
        float[] configValues = new float[]{-1.0f, 0.2f, 0.86f, 1.05f};
        boolean[] expectedOutcomes = new boolean[]{false, false, true, false};
        ImmutableMap configMappings = ImmutableMap.of((Object)"hbase.bucketcache.acceptfactor", (Object)configValues);
        Configuration conf = HBaseConfiguration.create();
        this.checkConfigValues(conf, (Map<String, float[]>)configMappings, expectedOutcomes);
    }

    @Test
    public void testInvalidMinFactorConfig() throws IOException {
        float[] configValues = new float[]{-1.0f, 0.0f, 0.96f, 1.05f};
        boolean[] expectedOutcomes = new boolean[]{false, true, false, false};
        ImmutableMap configMappings = ImmutableMap.of((Object)"hbase.bucketcache.minfactor", (Object)configValues);
        Configuration conf = HBaseConfiguration.create();
        this.checkConfigValues(conf, (Map<String, float[]>)configMappings, expectedOutcomes);
    }

    @Test
    public void testInvalidExtraFreeFactorConfig() throws IOException {
        float[] configValues = new float[]{-1.0f, 0.0f, 0.2f, 1.05f};
        boolean[] expectedOutcomes = new boolean[]{false, true, true, true};
        ImmutableMap configMappings = ImmutableMap.of((Object)"hbase.bucketcache.extrafreefactor", (Object)configValues);
        Configuration conf = HBaseConfiguration.create();
        this.checkConfigValues(conf, (Map<String, float[]>)configMappings, expectedOutcomes);
    }

    @Test
    public void testInvalidCacheSplitFactorConfig() throws IOException {
        float[] singleFactorConfigValues = new float[]{0.2f, 0.0f, -0.2f, 1.0f};
        float[] multiFactorConfigValues = new float[]{0.4f, 0.0f, 1.0f, 0.05f};
        float[] memoryFactorConfigValues = new float[]{0.4f, 0.0f, 0.2f, 0.5f};
        boolean[] expectedOutcomes = new boolean[]{true, false, false, false};
        ImmutableMap configMappings = ImmutableMap.of((Object)"hbase.bucketcache.single.factor", (Object)singleFactorConfigValues, (Object)"hbase.bucketcache.multi.factor", (Object)multiFactorConfigValues, (Object)"hbase.bucketcache.memory.factor", (Object)memoryFactorConfigValues);
        Configuration conf = HBaseConfiguration.create();
        this.checkConfigValues(conf, (Map<String, float[]>)configMappings, expectedOutcomes);
    }

    private void checkConfigValues(Configuration conf, Map<String, float[]> configMap, boolean[] expectSuccess) throws IOException {
        Set<String> configNames = configMap.keySet();
        for (int i = 0; i < expectSuccess.length; ++i) {
            try {
                for (String configName : configNames) {
                    conf.setFloat(configName, configMap.get(configName)[i]);
                }
                BucketCache cache = new BucketCache(this.ioEngineName, 0x2000000L, this.constructedBlockSize, this.constructedBlockSizes, 3, 64, this.persistencePath, 100, conf);
                Assert.assertTrue((String)("Created BucketCache and expected it to succeed: " + expectSuccess[i] + ", but it actually was: " + !expectSuccess[i]), (boolean)expectSuccess[i]);
                continue;
            }
            catch (IllegalArgumentException e) {
                Assert.assertFalse((String)("Created BucketCache and expected it to succeed: " + expectSuccess[i] + ", but it actually was: " + !expectSuccess[i]), (boolean)expectSuccess[i]);
            }
        }
    }

    private void validateGetPartitionSize(BucketCache bucketCache, float partitionFactor, float minFactor) {
        long expectedOutput = (long)Math.floor((float)bucketCache.getAllocator().getTotalSize() * partitionFactor * minFactor);
        Assert.assertEquals((long)expectedOutput, (long)bucketCache.getPartitionSize(partitionFactor));
    }

    @Test
    public void testOffsetProducesPositiveOutput() {
        long testValue = 549888460800L;
        BucketCache.BucketEntry bucketEntry = new BucketCache.BucketEntry(testValue, 10, 10L, true);
        Assert.assertEquals((long)testValue, (long)bucketEntry.offset());
    }

    @Test
    public void testCacheBlockNextBlockMetadataMissing() throws Exception {
        int size = 100;
        int length = 33 + size;
        byte[] byteArr = new byte[length];
        ByteBuffer buf = ByteBuffer.wrap(byteArr, 0, size);
        HFileContext meta = new HFileContextBuilder().build();
        HFileBlock blockWithNextBlockMetadata = new HFileBlock(BlockType.DATA, size, size, -1L, buf, true, -1L, 52, -1, meta);
        HFileBlock blockWithoutNextBlockMetadata = new HFileBlock(BlockType.DATA, size, size, -1L, buf, true, -1L, -1, -1, meta);
        BlockCacheKey key = new BlockCacheKey("key1", 0L);
        ByteBuffer actualBuffer = ByteBuffer.allocate(length);
        ByteBuffer block1Buffer = ByteBuffer.allocate(length);
        ByteBuffer block2Buffer = ByteBuffer.allocate(length);
        blockWithNextBlockMetadata.serialize(block1Buffer, true);
        blockWithoutNextBlockMetadata.serialize(block2Buffer, true);
        CacheTestUtils.getBlockAndAssertEquals((BlockCache)this.cache, key, (Cacheable)blockWithNextBlockMetadata, actualBuffer, block1Buffer);
        this.waitUntilFlushedToBucket(this.cache, key);
        CacheTestUtils.getBlockAndAssertEquals((BlockCache)this.cache, key, (Cacheable)blockWithoutNextBlockMetadata, actualBuffer, block1Buffer);
        this.cache.evictBlock(key);
        Assert.assertNull((Object)this.cache.getBlock(key, false, false, false));
        CacheTestUtils.getBlockAndAssertEquals((BlockCache)this.cache, key, (Cacheable)blockWithoutNextBlockMetadata, actualBuffer, block2Buffer);
        this.waitUntilFlushedToBucket(this.cache, key);
        CacheTestUtils.getBlockAndAssertEquals((BlockCache)this.cache, key, (Cacheable)blockWithNextBlockMetadata, actualBuffer, block1Buffer);
    }

    private static class MockedBucketCache
    extends BucketCache {
        public MockedBucketCache(String ioEngineName, long capacity, int blockSize, int[] bucketSizes, int writerThreads, int writerQLen, String persistencePath) throws FileNotFoundException, IOException {
            super(ioEngineName, capacity, blockSize, bucketSizes, writerThreads, writerQLen, persistencePath);
            this.wait_when_cache = true;
        }

        public void cacheBlock(BlockCacheKey cacheKey, Cacheable buf, boolean inMemory) {
            super.cacheBlock(cacheKey, buf, inMemory);
        }

        public void cacheBlock(BlockCacheKey cacheKey, Cacheable buf) {
            super.cacheBlock(cacheKey, buf);
        }
    }
}

